#ifndef BITMAP_HPP
#define BITMAP_HPP
#include "types.h" //for <windows.h>
#include "point4.hpp" //for Point class.
#include "rect1.hpp" //for Rect class.
/* The WindowBitmap class is used to maintain a bitmap copy of your
window for restoring it.  WindowBitmap uses a constructor that
takes an HWND as param.  To declare it globally, put at the top a global:
WindowBitmap *wbm;
then in WM_CREATE: put
	wbm = new WindowBitmap(hwnd, backgroundcolor);
backgroundcolor should be RGB(0,0,0) or RGB(255,255,255), depending on
whether your wndclass.hbrBackground is BLACK_BRUSH or WHITE_BRUSH.
In WM_PAINT: put
	hdc = BeginPaint(hwnd, &ps);
	wbm->CopyTo(hdc, ps.rcPaint);
	//Possible other screen writing
	EndPaint(hwnd, &ps);	wbm->Repaint(hwnd);
Alternately, if you don't write anything extra, in WM_PAINT, you can just put
	wbm->Repaint(hwnd);
In WM_DESTROY put
	delete wbm;
and to CLEAR the window put
	wbm->Clear();
	InvalidateRect(hwnd, NULL, FALSE);
Use the standard Windows API GUI calls to write graphics, except use
wbm->hdc() as the HDC argument.  Like, for instance,
	SetPixel(wbm->hdc(), x, y, col);
Whenever you finish a message case which does these writes to
the WindowBitmap, you do a
	InvalidateRect(hwnd, NULL, FALSE);
to BitBlt the altered WindowBitmap to the screen.

The WindowBitmap class can also be used for Copy, Cut and Paste.
Use the CutFrom method to get an image into the WindowBitmap and use
PasteTo to put it back out.
The TransparentWindowBitmap class is for showing images with a
"transparent" background.
*/

class TransparentWindowBitmap; /* This "forward" declaration is needed so
	WindowBitmap knows there will be such a class, so that it's _copy
	function and copy constructor know its ok to have a
	TransparentWindowBitmap type as an argument. */

class WindowBitmap
{
protected:
	HWND _hwnd;
	HDC _hdc;
	HBITMAP _hbitmap;
	COLORREF _blankcolor;
	HBRUSH _hblankbrush;
	int _CX, _CY;
	virtual void _allocate();
	virtual void _free();
	virtual void _copy(const WindowBitmap &wbm);
	virtual void _copy(const TransparentWindowBitmap &twbm);
public:
	//Constructors and destructor
	WindowBitmap(); //Default.
	WindowBitmap(HWND hwnd, COLORREF blankcol = RGB(255,255,255));
		//construct a bitmap the size of the screen.  You can specify
		//the blankcolor.
	WindowBitmap(HWND hwnd, int cx = 1, int cy = 1, COLORREF blankcol =
		RGB(255,255,255));//construct a bitmap of set size.
	/* Any class which does memory allocation in its constructors needs
		a copy constructor, as a term by term copy would not be appropriate;
		an HBITMAP might still be used in one copy when already Deleted by
		another copy's destructor */
	WindowBitmap(const WindowBitmap &wbm);
	WindowBitmap(const TransparentWindowBitmap &twbm);
	virtual ~WindowBitmap(); /* The destructor is delcared to be virtual
		because WindowBitmap has a child class TransparentWindowBitmap,
		and the child's destructor is different. The methods that differ
		between parent and child are also declared virtual.  */
	//Accessors
	HDC hdc(){return _hdc;}
	int cx(){return _CX;}
	int cy(){return _CY;}
	//Mutators
	void Clear(); //fill with _hblankbrush.
	virtual void SetBlankColor(COLORREF blankcol);
	virtual void CutFrom(HDC hdc, const Rect &rect);
	virtual void LoadResourceBitmap(HINSTANCE hInst, char * resource_name);
	virtual void LoadResourceBitmap(HINSTANCE hInst, int resource_ID);
	virtual void LoadFileBitmap(LPSTR file_bitmap_name, BOOL resizeflag = TRUE);
		/* resizeflag says whether or not change _CX and _CY to match the
		dimensions of the bitmap you are loading.*/
	virtual void SaveFileBitmap(LPSTR file_bitmap_name);
	//Methods
	virtual void CopyTo(HDC hdc, const Rect &rect);  //Our Rect class.
	virtual void PasteTo(HDC hdc, const Point &point);
	void StretchTo(HDC hdc, const Rect &rect);//Stretch or Shrink whole to fit rect.
	void Repaint(HWND hwnd);
	void CopyTo(HDC hdc, const RECT &rect);  //Windows RECT struct.
	WindowBitmap& operator=(const WindowBitmap &wbm);
	WindowBitmap& operator=(const TransparentWindowBitmap &wbm);
};

class TransparentWindowBitmap : public WindowBitmap
{
private:
	HDC _hdc_mask;
	HBITMAP _hbitmap_mask;
	BOOL _transparent_equals_corner_color;
	virtual void _allocate();
	void _allocate_child_fields();
	virtual void _free();
	void _free_child_fields();
	virtual void _copy(const WindowBitmap &wbm);
	virtual void _copy(const TransparentWindowBitmap &twbm);
	friend WindowBitmap; //for WindowBitmap::_copy(TransparentWindowBitmap&)
public:
	//Constructors and destructor
	TransparentWindowBitmap(); //Default.
	TransparentWindowBitmap(HWND hwnd, int cx = 1, int cy = 1, COLORREF transparent =
		RGB(255,255,255));/* constructs two cx x cy bitmaps, and uses transparent
		as the blankcol called by Clear() */
	TransparentWindowBitmap(const WindowBitmap &wbm);
	TransparentWindowBitmap(const TransparentWindowBitmap &twbm);
	virtual ~TransparentWindowBitmap(); /* Call this virtual to match
		the "virtual" on the parent calsses desctructor.  We put virtual
		in front of the methods which are virtual in the parent as well.
		Strictly speaking you don't need to use the "virtual" declaration
		in the child class, it only has an effect in when used in the
		parent class, but it's good practice to match the declarations. */
	//Accessors are the same, plusthis one new one.
	COLORREF transparent(){return _blankcolor;}
	/* All the Mutators except Clear need to be changed, and SetTransparent
		is new.  Note that Clear will fill *only* the _hdc with the
		_transparent color; to then make the bitmap truly transparent you
		would need to call Fixup(). */
	void SetTransparentEqualsCornerColor(BOOL onoff)
		{_transparent_equals_corner_color = onoff;}
	virtual void SetBlankColor(COLORREF transparent); //sets  _transparent.
	virtual void CutFrom(HDC hdc,const  Rect &rect);
	virtual void LoadResourceBitmap(HINSTANCE hInst, char * resource_name);
	virtual void LoadResourceBitmap(HINSTANCE hInst, int resource_ID);
	virtual void LoadFileBitmap(LPSTR file_bitmap_name,
 		BOOL resizeflag = TRUE);
	virtual void SaveFileBitmap(LPSTR file_bitmap_name);
	//Only these Methods need to change, the other two are derived.
	virtual void CopyTo(HDC hdc, const Rect &rect);  //Our Rect class.
	virtual void PasteTo(HDC hdc, const Point &point);
	//virtual void StretchTo(HDC hdc, const Rect &rect);//Stretch or Shrink whole to fit rect.
	//Finally, this is a new method.
	void Fixup(); /* Function used to convert a WindowBitmap style
		_hdc into a TransparentWindowBitmap pair of _hdc and _hdc_mask.
		Mostly this is used privately in the implementation of the methods,
		but you might want to call it if you'd drawn something in _hdc
		by calling Clear and then doing GDI calls. */
	TransparentWindowBitmap& operator=(const WindowBitmap &wbm);
	TransparentWindowBitmap& operator=(const TransparentWindowBitmap &wbm);
};
#endif //BITMAP_HPP
